angular.module("app", [
        'ngRoute',
        'ngResource',
        'ngSanitize',
        'app.services',
        'ngAnimate',
        'LocalStorageModule'
    ])
    .config($routeProvider =>
        $routeProvider
        .when('/', {
            templateUrl: "home.html",
            controller: "HomeController"
        }).when('', {
            templateUrl: "home.html",
            controller: "HomeController"
        }).when('/search', {
            templateUrl: "search.html",
            controller: "SearchController"
        }).when('/e/:entryId*', {
            templateUrl: "entry.html",
            controller: "EntryController"
        }).when('/info', {
            templateUrl: "info.html",
            controller: "InfoController"
        }).when('/internal', {
            templateUrl: "internal.html",
            controller: "InternalController"
        }).otherwise({
            redirectTo: "/"
        })
    );
angular.module('app.services', []);


// Replace broken images with default one
brokenImage = img => {
    img.src = "/images/404.svg";
};

angular
    .module("app.services")
    .factory("JsonData", ["$resource", "$cacheFactory", function($resource, $cacheFactory) {
        const localCache = $cacheFactory('JsonData');

        // Angular resource to fetch files inside /data/
        const res = $resource("data/:fileName", {
            fileName: "@fileName"
        }, {
            query: {
                isArray: true,
                cache: localCache
            }
        });

        res.getData = function(cb) {
            const me = this;
            me.query({
                fileName: "data.json"
            }, data => {
                delete data.$promise;
                delete data.$resolved;
                cb(data);
            });
        };

        return res;
    }])

    .factory("JsonConfig", ["$resource", "$cacheFactory", function($resource, $cacheFactory) {
        const localCache = $cacheFactory('JsonConfig');

        // Angular resource to fetch files inside /data/
        const res = $resource("data/:fileName", {
            fileName: "@fileName"
        }, {
            query: {
                isArray: false,
                cache: localCache
            }
        });

        res.getConfig = function(cb) {
            const me = this;
            me.query({
                fileName: "config.json"
            }, data => {
                delete data.$promise;
                delete data.$resolved;
                cb(data);
            });
        };

        return res;
    }]);

angular
    .module("app.services")
    .filter("jsonFilter", function() {
        return function(input, filterObject) {
            if (input == undefined) {
                return [];
            }

            // Search Strings
            const searchString = function(entry) {
                let searchString = filterObject.searchString;
                // No String given
                if (searchString === "") {
                    return true;
                }
                // String in label?
                if (entry.name &&
                    entry.name.toLowerCase().indexOf(searchString.toLowerCase()) >= 0) {
                    return true;
                }
                return false;
            };

            // Search applied tags
            const searchTags = function(element) {
                if (filterObject.tags.length === 0) {
                    return true;
                }
                if (! element.tags) {
                    return false;
                }
                for (tagName of filterObject.tags) {
                    if (! element.tags.includes(tagName)) {
                        return false;
                    }
                }
                return true;
            };

            const searchHandicappedSuited = function(element) {
                if (element.accessibility == null) {
                    element.accessibility = 0;
                }
                return filterObject.accessibility <= element.accessibility;
            };

            return input.filter(searchString)
                .filter(searchTags)
                .filter(searchHandicappedSuited);
        };
    });

angular
    .module("app")
    .controller("EntryController",
        ["$scope", "$routeParams", "JsonData", "JsonConfig", "$anchorScroll", "$timeout", function($scope,
            $routeParams,
            JsonData,
            JsonConfig,
            $anchorScroll,
            $timeout) {

            let loadMap = () => {
                if (!$scope.validMap) {
                    return;
                }
                // Only create the map if it does not exist already
                if ($scope.map) {
                    return;
                }
                // LatLong position of the entry
                let entryPosition = [
                    $scope.entry.gps.lat,
                    $scope.entry.gps.long
                ];
                $scope.map = L.map('map').setView(entryPosition, 13);
                // Setup map
                L.tileLayer('https://api.tiles.mapbox.com/v4/{id}/{z}/{x}/{y}.png?access_token={accessToken}', {
                    attribution: 'Map data &copy; <a href="https://www.openstreetmap.org/">OpenStreetMap</a> contributors, <a href="https://creativecommons.org/licenses/by-sa/2.0/">CC-BY-SA</a>, Imagery © <a href="https://www.mapbox.com/">Mapbox</a>',
                    maxZoom: 18,
                    id: 'mapbox.streets',
                    accessToken: 'pk.eyJ1IjoibWVnYW1hbm1hbHRlIiwiYSI6ImNqbDR5bG1rOTFpc2wzdnF5eTJ0dnhtemsifQ.RCMTLOjQri0YrwcqeDXqeA'
                }).addTo($scope.map);
                // Add the marker to the map
                let marker = L.marker(entryPosition);
                // Add a nice uncloseable popup
                marker.bindPopup(`<b>${$scope.entry.name}</b>`, {
                    closeButton: false,
                    autoClose: false,
                    closeOnEscapeKey: false,
                    closeOnClick: false
                });
                marker.addTo($scope.map);
                // Bad way of handling container size changes due to ng-show
                // TODO Improve
                setTimeout(() => {
                    $scope.map.invalidateSize();
                    marker.openPopup();
                }, 100);
            };

            let tags = [];
            $scope.iconFromTag = tagName => {
                let matching = tags.filter(tag => tag.name == tagName);
                if (matching.length > 0) {
                    return matching[0].icon;
                }
                return "fa-question";
            };

            // Accessibility to traffic light color
            $scope.accessibilityToColor = accessibility => {
                if (accessibility === 1) {
                    return 'orange';
                } else if (accessibility === 2) {
                    return 'green';
                }
                return 'red';
            };

            // Accessibility to description
            $scope.accessibilityToText = accessibility => {
                if (accessibility === 1) {
                    return 'Eingeschränkt Barrierefrei';
                } else if (accessibility === 2) {
                    return 'Barrierefrei';
                } else {
                    return 'Nicht Barrierefrei';
                }
            };

            // Fetch the data and filter for the currently viewed entry
            JsonData.getData(function(data) {
                $scope.data = data;
                $scope.data
                    .filter(el => el.id.toString() === $routeParams.entryId)
                    .forEach(el => {
                        // Expose extend object
                        el = Object.assign(el, el.extend);
                        delete el.extend;
                        $scope.entry = el;
                    });

                // Are we able to display a map correctly?
                $scope.validMap =
                    $scope.entry.gps != null &&
                    $scope.entry.gps.lat != null &&
                    $scope.entry.gps.long != null;
                loadMap();
            });

            // Fetch the configuration for the tag icons
            JsonConfig.getConfig(config => {
                tags = config.tags;
            });

            $anchorScroll();
        }]);

angular
    .module("app")
    .controller("HomeController", ["$scope", "JsonData", function($scope, JsonData) {
        // Fetch data to count the number of entries
        JsonData.getData(data => $scope.baseData = data);
    }]);

angular
    .module("app")
    .controller("InfoController", ["$scope", "JsonData", function($scope, JsonData) {
        // Nothing todo?
    }]);

angular
    .module("app")
    .controller("InternalController",
        ["$scope", "JsonData", "$timeout", function($scope,
            JsonData,
            $timeout) {

            JsonData.getData(data => $scope.rawJson = data);

            $scope.select = which => {
                $scope.selected = which;
            };
            $scope.selected = 'table';
        }]);

angular
    .module("app")
    .controller("NaviController",
        ["$scope", "$routeParams", "$controller", "$rootScope", "$location", function($scope,
            $routeParams,
            $controller,
            $rootScope,
            $location) {

            $scope.activeMenu = "";

            $scope.$on('$routeChangeSuccess', function(next, current) {
                $scope.activeMenu = "";
                if (current.$$route.originalPath.indexOf("/info") >= 0) { // info
                    $scope.activeMenu = "info";
                } else if (current.$$route.originalPath.indexOf("/search") >= 0) { // search
                    $scope.activeMenu = "search";
                } else if (current.$$route.originalPath.indexOf("/e") >= 0) { // search
                    $scope.activeMenu = "detail";
                } else {
                    $scope.activeMenu = "";
                }
            });
        }]);

angular
    .module("app")
    .controller("SearchController",
        ["$scope", "$routeParams", "JsonData", "JsonConfig", "$timeout", "$templateRequest", "$compile", "$filter", "localStorageService", function($scope,
            $routeParams,
            JsonData,
            JsonConfig,
            $timeout,
            $templateRequest,
            $compile,
            $filter,
            localStorageService) {

            // Reset filter function
            $scope.resetFilter = () => $scope.option = {
                tags: [],
                searchString: "",
                accessibility: 0
            };

            $scope.toggleTag = (tag) => {
                let index = $scope.option.tags.findIndex(
                    el => el == tag.name);
                if (index >= 0) {
                    $scope.option.tags.splice(index, 1);
                } else {
                    $scope.option.tags.push(tag.name);
                }
            };

            $scope.loadMap = () => {
                if ($scope.map != null) {
                    return;
                }
                $scope.map = L.map('map-all').setView([51.505, -0.09], 13);
                // Setup map
                L.tileLayer('https://api.tiles.mapbox.com/v4/{id}/{z}/{x}/{y}.png?access_token={accessToken}', {
                    attribution: 'Map data &copy; <a href="https://www.openstreetmap.org/">OpenStreetMap</a> contributors, <a href="https://creativecommons.org/licenses/by-sa/2.0/">CC-BY-SA</a>, Imagery © <a href="https://www.mapbox.com/">Mapbox</a>',
                    maxZoom: 18,
                    id: 'mapbox.streets',
                    accessToken: 'pk.eyJ1IjoibWVnYW1hbm1hbHRlIiwiYSI6ImNqbDR5bG1rOTFpc2wzdnF5eTJ0dnhtemsifQ.RCMTLOjQri0YrwcqeDXqeA'
                }).addTo($scope.map);
                // Update all markers
                $scope.updateMarkers();
            };

            let updateTagCount = () => {
                $scope.tagCount = {};
                $scope.filteredData.forEach(entry => {
                    if (! entry.tags) {
                        return;
                    }
                    entry.tags.forEach(tag => {
                        if (!$scope.tagCount[tag]) {
                            $scope.tagCount[tag] = 1;
                        } else {
                            $scope.tagCount[tag] += 1;
                        }
                    });
                });
            };

            // Handle tab selection
            $scope.select = function(which) {
                // Prevent bad map loading
                if (which === 'map' && $scope.filteredData.length === 0) {
                    return;
                }
                $scope.selected = which;

                if (which === "map") {
                    $scope.loadMap();
                    // Timeout to fix false loading
                    // TODO: Improve
                    $timeout(() => {
                        if ($scope.markerGroup.getLayers().length != 0) {
                            $scope.map.invalidateSize();
                            $scope.map.fitBounds($scope.markerGroup.getBounds().pad(0.3));
                        }
                    }, 300);
                }
            };

            // Default is list view
            $scope.selected = 'list';

            // Update map markers
            $scope.updateMarkers = function() {
                $scope.markers = [];
                if ($scope.filteredData == null) {
                    return;
                }
                // Add all entries that have coordinates to the list of markers
                for (let entry of $scope.filteredData) {
                    let validLocation =
                        entry['gps'] != null &&
                        entry['gps']['lat'] != null &&
                        entry['gps']['long'] != null;
                    if (validLocation) {
                        $scope.markers.push(entry);
                    }
                }
                // Delete old markers
                if ($scope.markerGroup != null &&
                    $scope.map != null) {
                    $scope.map.removeLayer($scope.markerGroup);
                }
                // Create a layer group for all markers
                $scope.markerGroup = L.markerClusterGroup({
                    zoomToBoundsOnClick: false
                });
                // Actually create and add the markers to the markergroup
                for (let marker of $scope.markers) {
                    createMarker(marker).addTo($scope.markerGroup);
                }
                // Add event listener for clicks on clusters
                $scope.markerGroup.on('clusterclick', (a) => {
                    $scope.map.fitBounds(a.layer.getBounds().pad(0.1));
                });
                // Add the markergroup to the map
                if ($scope.map != null) {
                    $scope.markerGroup.addTo($scope.map);
                }
            };

            // Create a marker from the given entry
            let createMarker = entry => {
                let pos = [
                    entry.gps.lat,
                    entry.gps.long
                ];
                let m = L.marker(pos);
                let id = encodeURIComponent(entry.id);
                let img = "";
                if (entry.imgs && entry.imgs.length > 0) {
                    img = entry.imgs[0];
                }
                let template = `
<div class="card" style="width: 14rem;">
  <img class="card-img-top" src="${img}" alt="" onerror="brokenImage(this)">
  <div class="card-body">
    <h5 class="card-title">${entry.name}</h5>
  </div>
  <div class="card-footer">
    <a class="btn btn-block btn-primary btn-sm text-light" href="#!/e/${id}">Mehr Infos</a>
  </div>
</div>`;
                let popup = L.popup({
                        closeButton: false
                    })
                    .setContent(template);
                m.bindPopup(popup);
                m.update();
                return m;
            };

            $scope.updateFilter = function() {
                $scope.filteredData = $filter('jsonFilter')($scope.data, $scope.option);
                $scope.updateMarkers();
                updateTagCount();
            };

            JsonData.getData(function(data) {
                $scope.data = data;
                $scope.updateFilter();
            });

            JsonConfig.getConfig(function(config) {
                $scope.config = config;
            });

            $scope.$watchCollection("option.tags", function(newValue, oldValue) {
                $scope.updateFilter();
                localStorageService.set("optionSave", $scope.option);
            });
            $scope.$watchCollection("option", function(newValue, oldValue) {
                $scope.updateFilter();
                localStorageService.set("optionSave", $scope.option);
            });

            $scope.resetFilter();
            // Initialize search options
            $scope.option = localStorageService.get("optionSave");
            if (($scope.option == null)) {
                $scope.resetFilter();
            }
        }]);

//# sourceMappingURL=data:application/json;charset=utf8;base64,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
